function [fig, t] = mainProfiles(prof, veh, profNames, xaxis)
arguments
    prof
    veh struct {mustBeA(veh, "struct")} = [];
    profNames {mustBeText} = ["def"];
    xaxis = "time"
end
% mainProfiles(prof)
%  draw post-processing plots.
%
% Input arguments
% ---------------
% prof : struct
%   data structures for the time profiles.
% veh : struct
%   vehicle data.
% profNames (optional) : string 
%   specify the time profiles to be plotted.

if strcmp(profNames(1), "def")
    profNames = ["vehSpd", "vehAcc"];
    if isfield(prof, "engTrq")
        profNames = [profNames, "gearNumber", "engSpd", "engTrq", "unfeas"];
    end
    if isfield(prof, "emTrq")
        profNames = [profNames, "gearNumber", "emSpd", "emTrq", "unfeas"];
    end
    if isfield(prof, "battSOC")
        profNames = [profNames, "battSOC", "battCurr"];
    end
end

%% Load info
time = 0:1:(length([prof.time])-1);
time = time(:);

switch xaxis
    case "time"
        xaxis = [prof.time];
        xAxisLabel = "Time, s";
    case "dist"
        xaxis = [prof.vehDist];
        xAxisLabel = "Distance, m";
end

%% Plot
lbl = labels();
clr = colors();

fig = figure;
t = tiledlayout("flow");
for n = 1:length(profNames)
    if isfield(prof, profNames(n))
        ax(n) = nexttile;

        % Line color
        if isfield(clr, profNames(n))
            color = clr.(profNames(n));
        else
            color = "#7f7f7f";
        end

        plot(xaxis, [prof.(profNames(n))], 'LineWidth', 1.5, 'Color', color)
        grid on
        hold on

        % Bounds
        [ubProf, lbProf] = profBounds(profNames(n), prof, veh);
        if ~isempty(ubProf)
            plot(xaxis, ubProf, '--', 'Color', [0.8 0.8 0.8])
        end
        if ~isempty(lbProf)
            plot(xaxis, lbProf, '--', 'Color', [0.8 0.8 0.8])
        end

        % Labels
        if isfield(lbl, profNames(n))
            ylabel(lbl.(profNames(n)))
        else
            ylabel(profNames(n))
        end

    else
        warning(profNames(n) + " not found.")
    end
end

xlabel(t, xAxisLabel, 'FontSize', 12)
linkaxes(ax, 'x')

end

function lbl = labels()
    % Vehicle
    lbl.vehSpd = "Vehicle speed, m/s";
    lbl.vehAcc = "Vehicle acceleration, m/s";
    lbl.vehSlope = "Road slope, rad";
    lbl.rollForce = "Rolling resistance, N";
    lbl.gradeForce = "Slope resistance, N";
    lbl.dragForce = "Air drag resistance, N";
    lbl.vehForce = "Total resistance, N";
    % Wheel
    lbl.wheelSpd = "Wheel speed, rad/s";
    lbl.wheelTrq = "Wheel torque, Nm";
    % Brakes
    lbl.brakeCmd = "Brake command, -";
    lbl.brakeTrq = "Brakes torque, Nm";
    lbl.brakeUnfeas = "Brakes unfeas, -";
    % Final drive
    lbl.fdSpd = "Final drive speed, rad/s";
    lbl.fdTrq = "Final drive torque, Nm";
    % Transmission
    lbl.gbSpd = "Transmission speed, rad/s";
    lbl.gbTrq = "Transmission torque, Nm";
    lbl.gearNumber = "Gear number, -";
    % Engine
    lbl.engSpd = "Engine speed, rad/s";
    lbl.engTrq = "Engine torque, Nm";
    lbl.fuelFlwRate = "Fuel flow rate, kg/s";
    lbl.noxFlwRate = "NOx flow rate, kg/s";
    lbl.exhFlwRate = "Exhaust gases flow rate, kg/s";
    lbl.exhTemp = "Exhaust gases temperature, K";
    lbl.engSpdUnfeas = "Engine speed unfeas, -";
    lbl.engTrqUnfeas = "Engine torque unfeas, -";
    % EM
    lbl.emSpd = "E-machine speed, rad/s";
    lbl.emTrq = "E-machine torque, Nm";
    lbl.emElPwr = "E-machine electrical power, W";
    lbl.emSpdUnfeas = "EM speed unfeas, -";
    lbl.emTrqUnfeas = "EM torque unfeas, -";
    % Battery
    lbl.battCurr = "Battery current, A";
    lbl.battVolt = "Battery voltage, V";
    lbl.battEqRes = "Battery int. resistance, \ohm";
    lbl.battOCVolt = "Battery OCV, V";
    lbl.battSOC = "Battery SOC, -";
    lbl.battUnfeas = "Battery unfeas, -";
    lbl.battAhThroughput = "Battery Ah throughput, Ah";
    lbl.battPower = "Battery power, W";
    lbl.battEnergy = "Battery energy, J";
    % Others
    lbl.unfeas = "Total unfeas, -";
    lbl.refSpd = "Mission speed, m/s";
end

function clr = colors()
    % Vehicle
    clr.vehSpd = "#2ca02c";
    clr.vehAcc = "#d62728";
    clr.vehSlope = "#8c564b";
    clr.rollForce = "#ff7f0e";
    clr.gradeForce = "#9467bd";
    clr.dragForce = "#17becf";
    clr.vehForce = "#d62728";
    % Wheel
    clr.wheelSpd = "#2ca02c";
    clr.wheelTrq = "#d62728";
    % Brakes
    clr.brakeCmd = "#ff7f0e";
    clr.brakeTrq = "#d62728";
    clr.brakeUnfeas = "#7f7f7f";
    % Final drive
    clr.fdSpd = "#2ca02c";
    clr.fdTrq = "#d62728";
    % Transmission
    clr.gbSpd = "#2ca02c";
    clr.gbTrq = "#d62728";
    clr.gearNumber = "#ff7f0e";
    % Engine
    clr.engSpd = "#2ca02c";
    clr.engTrq = "#d62728";
    clr.fuelFlwRate = "#ff7f0e";
    clr.noxFlwRate = "NOx flow rate, kg/s";
    clr.exhFlwRate = "#9467bd";
    clr.exhTemp = "#d62728";
    clr.engSpdUnfeas = "#7f7f7f";
    clr.engTrqUnfeas = "#7f7f7f";
    % EM
    clr.emSpd = "#2ca02c";
    clr.emTrq = "#d62728";
    clr.emElPwr = "#1f77b4";
    clr.emSpdUnfeas = "#7f7f7f";
    clr.emTrqUnfeas = "#7f7f7f";
    % Battery
    clr.battCurr = "#d62728";
    clr.battVolt = "#2ca02c";
    clr.battEqRes = "#ff7f0e";
    clr.battOCVolt = "#9467bd";
    clr.battSOC = "#17becf";
    clr.battUnfeas = "#7f7f7f";
    clr.battAhThroughput = "#2ca02c";
    clr.battPower = "#ff7f0e";
    clr.battEnergy = "#9467bd";
    % Others
    clr.unfeas = "#7f7f7f";
    clr.refSpd = "#1f77b4";
end

function [ubProf, lbProf] = profBounds(profName, prof, veh)

switch profName
    % Vehicle
    case "vehSpd"
        if ~isempty(veh)
            ubProf = veh.maxSpd .* ones(1,length(prof));
        else
            ubProf = [];
        end
        lbProf = [];
    % Engine
    case "engSpd"
        if ~isempty(veh)
            ubProf = veh.eng.maxSpd .* ones(1,length(prof));
            lbProf = veh.eng.idleSpd .* ones(1,length(prof));
        else
            ubProf = [];
            lbProf = [];
        end
    case "engTrq"
        ubProf = [prof.engMaxTrq];
        if isfield(prof, "motTrq")
            lbProf = [prof.engMotTrq];
        else
            lbProf = zeros .* ones(1,length(prof));
        end
    % EM
    case "emSpd"
        if ~isempty(veh)
            ubProf = veh.em.maxSpd .* ones(1,length(prof));
        else
            ubProf = [];
        end
        lbProf = zeros .* ones(1,length(prof));
    case "emTrq"
        ubProf = [prof.emMaxTrq];
        lbProf = [prof.emMinTrq];
    % Battery
    case "battCurr"
        if ~isempty(veh)
            ubProf = veh.batt.maxCurr .* ones(1,length(prof));
            lbProf = veh.batt.minCurr .* ones(1,length(prof));
        else
            ubProf = [];
            lbProf = [];
        end
    case "battSOC"
        if ~isempty(veh)
            ubProf = veh.batt.SOC_ub .* ones(1,length(prof));
            lbProf = veh.batt.SOC_lb .* ones(1,length(prof));
        else
            ubProf = [];
            lbProf = [];
        end

    otherwise
        ubProf = [];
        lbProf = [];
end

end